import axios from 'axios';
import difyConfig from '../config';
import { useAppStore } from '../store/app';

// 创建 axios 实例
const request = axios.create({
    baseURL: `${difyConfig.DIFY_URL}`, // 替换为你的 API 基础地址
    timeout: 10000, // 设置超时时间
    headers: {
        'Content-Type': 'application/json',
    },
});

// 请求拦截器
request.interceptors.request.use(
    (config) => {
        const appStore = useAppStore()
        // 在发送请求之前做些什么，例如添加 token
        if (difyConfig.API_KEY) {
            config.headers.Authorization = `Bearer ${difyConfig.API_KEY}`;
        }
        // 当存在params参数时，主动加入userID
        if (config.params) {
            config.params.user = appStore.userId || difyConfig.USER_ID
        }
        // 当存在data参数时，主动加入userID
        if (config.data) {
            config.data.user = appStore.userId || difyConfig.USER_ID
        }
        return config;
    },
    (error) => {
        // 对请求错误做些什么
        return Promise.reject(error);
    }
);

// 响应拦截器
request.interceptors.response.use(
    (response) => {
        // 对响应数据做点什么，例如直接返回 data
        return response.data || response;
    },
    (error) => {
        // 统一处理错误
        const { response } = error;
        if (response) {
            // 服务器返回了状态码
            console.error(`Error ${response.status}: ${response.statusText}`);
            switch (response.status) {
                case 401:
                    console.error('未授权，请登录！');
                    break;
                case 403:
                    console.error('权限不足！');
                    break;
                case 404:
                    console.error('请求资源未找到！');
                    break;
                case 500:
                    console.error('服务器内部错误！');
                    break;
                default:
                    console.error(`未知错误：${response.statusText}`);
            }
        } else if (error.request) {
            // 请求已发出，但没有收到响应
            console.error('请求未收到响应:', error.request);
        } else {
            // 发送请求时出错
            console.error('请求错误:', error.message);
        }
        return Promise.reject(error);
    }
);

// 封装常用的请求方法
const get = (url, params) => request.get(url, { params });
const post = (url, data, config={}) => request.post(url, data, config);
const put = (url, data, config={}) => request.put(url, data, config);
const deleteRequest = (url, params, data) => request.delete(url, { params, data });

const stream = async (url, method, data) => {
    try {
        const appStore = useAppStore()
        data.user = appStore.userId || difyConfig.USER_ID
        url = difyConfig.DIFY_URL + url
        const response = await fetch(url, {
            method,
            headers: {
                "Content-Type": 'application/json',
                "Authorization": `Bearer ${difyConfig.API_KEY}`
            },
            body: JSON.stringify(data)
        });
        return response
    }catch (error) {
        console.error('Error:', error);
    }
}

// 导出封装后的请求方法
export default {
    get,
    post,
    put,
    delete: deleteRequest,
    stream
};